/*
 * Copyright (c) 2017 SpaceToad and the BuildCraft team
 * This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0. If a copy of the MPL was not
 * distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/
 */

/*
 * Copyright (c) 2019 SpaceToad and the BuildCraft team
 * This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0. If a copy of the MPL was not
 * distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/
 */
package alexiil.mc.mod.pipes.pipe;

import java.util.EnumSet;

import javax.annotation.Nonnull;

import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NbtCompound;
import net.minecraft.network.RegistryByteBuf;
import net.minecraft.registry.RegistryWrapper;
import net.minecraft.util.DyeColor;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Direction;
import net.minecraft.util.math.Vec3d;

import alexiil.mc.lib.attributes.item.ItemStackUtil;

import alexiil.mc.mod.pipes.util.TagUtil;

import alexiil.mc.lib.net.IMsgReadCtx;
import alexiil.mc.lib.net.IMsgWriteCtx;
import alexiil.mc.lib.net.NetByteBuf;

public class TravellingItem {
    // Client fields - public for rendering
    // @Nonnull
    // public final Supplier<ItemStack> clientItemLink; -- optimisation not ported
//    public int stackSize;
    public DyeColor colour;

    // Server fields
    /** The server itemstack */
    @Nonnull
    public ItemStack stack;
    int id = 0;
    boolean toCenter;
    double speed = 0.05;
    /** Absolute times (relative to world.getTotalWorldTime()) with when an item started to when it finishes. */
    long tickStarted, tickFinished;
    /** Relative times (from tickStarted) until an event needs to be fired or this item needs changing. */
    int timeToDest;
    /** If {@link #toCenter} is true then this represents the side that the item is coming from, otherwise this
     * represents the side that the item is going to. */
    Direction side;
    /** A set of all the faces that this item has tried to go and failed. */
    EnumSet<Direction> tried = EnumSet.noneOf(Direction.class);
    /** If true then events won't be fired for this, and this item won't be dropped by the pipe. However it will affect
     * pipe.isEmpty and related gate triggers. */
    boolean isPhantom = false;

    // @formatter:off
    /* States (server side):
      
      - TO_CENTER:
        - tickStarted is the tick that the item entered the pipe (or bounced back)
        - tickFinished is the tick that the item will reach the center 
        - side is the side that the item came from
        - timeToDest is equal to timeFinished - timeStarted
      
      - TO_EXIT:
       - tickStarted is the tick that the item reached the center
       - tickFinished is the tick that the item will reach the end of a pipe 
       - side is the side that the item is going to 
       - timeToDest is equal to timeFinished - timeStarted. 
     */
    // @formatter:on

    public TravellingItem(@Nonnull ItemStack stack) {
        this.stack = stack;
    }

//    public TravellingItem(ItemStack clientStack, int count) {
//        this.stackSize = count;
//        this.stack = clientStack;
//    }

    public TravellingItem(NbtCompound nbt, RegistryWrapper.WrapperLookup lookup, long tickNow) {
        stack = ItemStackUtil.fromNbt(nbt.getCompound("stack"), lookup);
        int c = nbt.getByte("colour");
        this.colour = c == 0 ? null : DyeColor.byId(c - 1);
        this.toCenter = nbt.getBoolean("toCenter");
        this.speed = nbt.getDouble("speed");
        if (speed < 0.001) {
            // Just to make sure that we don't have an invalid speed
            speed = 0.001;
        }
        tickStarted = nbt.getInt("tickStarted") + tickNow;
        tickFinished = nbt.getInt("tickFinished") + tickNow;
        timeToDest = nbt.getInt("timeToDest");

        side = TagUtil.readEnum(nbt.get("side"), Direction.class);
        if (side == null || timeToDest == 0) {
            // Older 8.0.x. version
            toCenter = true;
        }
        tried = TagUtil.readEnumSet(nbt.get("tried"), Direction.class);
        isPhantom = nbt.getBoolean("isPhantom");
    }

    public TravellingItem(NetByteBuf buf, IMsgReadCtx ctx, long tickNow) {
        if (buf.readBoolean()) {
            stack = ItemStack.PACKET_CODEC.decode(new RegistryByteBuf(buf, ctx.getConnection().getPlayer().getRegistryManager()));
        } else {
            stack = ItemStack.EMPTY;
        }
        int c = buf.readByte();
        this.colour = c == 0 ? null : DyeColor.byId(c - 1);
        this.toCenter = buf.readBoolean();
        this.speed = buf.readDouble();
        if (speed < 0.001) {
            speed = 0.001;
        }
        tickStarted = buf.readVarUnsignedInt() + tickNow;
        tickFinished = buf.readVarUnsignedInt() + tickNow;
        timeToDest = buf.readVarUnsignedInt();

        side = buf.readEnumConstant(Direction.class);
        tried = buf.readEnumSet(Direction.class);
        isPhantom = buf.readBoolean();
    }

    public NbtCompound writeToNbt(RegistryWrapper.WrapperLookup lookup, long tickNow) {
        NbtCompound nbt = new NbtCompound();
        nbt.put("stack", ItemStackUtil.writeNbt(stack, lookup));
        nbt.putByte("colour", (byte) (colour == null ? 0 : colour.getId() + 1));
        nbt.putBoolean("toCenter", toCenter);
        nbt.putDouble("speed", speed);
        nbt.putInt("tickStarted", (int) (tickStarted - tickNow));
        nbt.putInt("tickFinished", (int) (tickFinished - tickNow));
        nbt.putInt("timeToDest", timeToDest);
        nbt.put("side", TagUtil.writeEnum(side));
        nbt.put("tried", TagUtil.writeEnumSet(tried, Direction.class));
        if (isPhantom) {
            nbt.putBoolean("isPhantom", true);
        }
        return nbt;
    }

    public void writeToBuffer(NetByteBuf buf, IMsgWriteCtx ctx, long tickNow) {
        if (stack.isEmpty()) {
            buf.writeBoolean(false);
        } else {
            buf.writeBoolean(true);
            ItemStack.PACKET_CODEC.encode(new RegistryByteBuf(buf, ctx.getConnection().getPlayer().getRegistryManager()), stack);
        }
        buf.writeByte((byte) (colour == null ? 0 : colour.getId() + 1));
        buf.writeBoolean(toCenter);
        buf.writeDouble(speed);
        buf.writeVarUnsignedInt((int) (tickStarted - tickNow));
        buf.writeVarUnsignedInt((int) (tickFinished - tickNow));
        buf.writeVarUnsignedInt(timeToDest);
        
        buf.writeEnumConstant(side);
        buf.writeEnumSet(tried, Direction.class);
        buf.writeBoolean(isPhantom);
    }

    public int getCurrentDelay(long tickNow) {
        long diff = tickFinished - tickNow;
        if (diff < 0) {
            return 0;
        } else {
            return (int) diff;
        }
    }

    public double getWayThrough(long now) {
        long diff = tickFinished - tickStarted;
        long nowDiff = now - tickStarted;
        return nowDiff / (double) diff;
    }

    public void genTimings(long now, double distance) {
        tickStarted = now;
        timeToDest = (int) Math.ceil(distance / speed);
        tickFinished = now + timeToDest;
    }

    public boolean canMerge(TravellingItem with) {
        if (isPhantom || with.isPhantom) {
            return false;
        }
        return toCenter == with.toCenter//
            && colour == with.colour//
            && side == with.side//
            && Math.abs(tickFinished - with.tickFinished) < 4//
            && stack.getMaxCount() >= stack.getCount() + with.stack.getCount()//
            && ItemStackUtil.areEqualIgnoreAmounts(stack, with.stack);
    }

    /** Attempts to merge the two travelling item's together, if they are close enough.
     * 
     * @param with
     * @return */
    public boolean mergeWith(TravellingItem with) {
        if (canMerge(with)) {
            this.stack.increment(with.stack.getCount());
            return true;
        }
        return false;
    }

    public Vec3d interpolatePosition(Vec3d start, Vec3d end, long tick, float partialTicks) {
        long diff = tickFinished - tickStarted;
        long nowDiff = tick - tickStarted;
        double sinceStart = nowDiff + partialTicks;
        double interpMul = sinceStart / diff;
        double oneMinus = 1 - interpMul;
        if (interpMul <= 0) return start;
        if (interpMul >= 1) return end;

        double x = oneMinus * start.x + interpMul * end.x;
        double y = oneMinus * start.y + interpMul * end.y;
        double z = oneMinus * start.z + interpMul * end.z;
        return new Vec3d(x, y, z);
    }

    public Vec3d getRenderPosition(BlockPos pos, long tick, float partialTicks, ISimplePipe pipe) {
        long diff = tickFinished - tickStarted;
        long afterTick = tick - tickStarted;

        float interp = (afterTick + partialTicks) / diff;
        interp = Math.max(0, Math.min(1, interp));

        Vec3d center = Vec3d.of(pos).add(0.5, 0.5, 0.5);
        Vec3d vecSide =
            side == null ? center : center.add(Vec3d.of(side.getVector()).multiply(pipe.getPipeLength(side)));

        Vec3d vecFrom;
        Vec3d vecTo;
        if (toCenter) {
            vecFrom = vecSide;
            vecTo = center;
        } else {
            vecFrom = center;
            vecTo = vecSide;
        }

        return vecFrom.multiply(1 - interp).add(vecTo.multiply(interp));
    }

    public Direction getRenderDirection(long tick, float partialTicks) {
        long diff = tickFinished - tickStarted;
        long afterTick = tick - tickStarted;

        float interp = (afterTick + partialTicks) / diff;
        interp = Math.max(0, Math.min(1, interp));
        if (toCenter) {
            return side == null ? null : side.getOpposite();
        } else {
            return side;
        }
    }

    public boolean isVisible() {
        return true;
    }
}
